#include "GUI/SimpleTest.h"
#include "Turing/Turing.h"
#include "Turing/TuringScanner.h"
#include "Turing/TuringParser.h"
#include "Grabbag/GrabbagTester.h"
#include <string>
#include <vector>
#include <utility>
#include <fstream>
using namespace std;

namespace {
    void runTests(const string& filename);
}

PROVIDED_TEST("MiddleA.tm") {
    runTests("MiddleA.tm");
}

PROVIDED_TEST("Power2.tm") {
    runTests("Power2.tm");
}

PROVIDED_TEST("Equal.tm") {
    runTests("Equal.tm");
}

namespace {
    const string kBaseDir = "res/";

    /* Maximum number of steps to run things for. */
    const int kMaxSteps = 10000000;

    bool readEntry(pair<string, bool>& entry, istream& in) {
        string first, second;
        if (!getline(in, first) || !getline(in, second)) return false;

        entry = make_pair(first, second == "true");
        return true;
    }

    void runTests(const string& filename) {
        ifstream input(kBaseDir + filename, ios::binary);
        if (!input) SHOW_ERROR("Error opening file " + filename);

        Turing::Program tm(input);
        if (!tm.isValid()) SHOW_ERROR("TM contains errors and cannot be run; use the debugger to see what they are.");

        runPrivateTest(filename, [&](istream& in) {
            for (pair<string, bool> entry; readEntry(entry, in); ) {
                Turing::Interpreter interpreter(tm, { entry.first.begin(), entry.first.end() });

                for (int i = 0; i <= kMaxSteps && interpreter.state() == Turing::Result::RUNNING; i++) {
                    interpreter.step();
                }

                if (interpreter.state() == Turing::Result::RUNNING) {
                    SHOW_ERROR("TM still running after " + addCommasTo(kMaxSteps) + " steps on input \"" + entry.first + "\". Possible infinite loop?");
                } else if (interpreter.state() == Turing::Result::ACCEPT) {
                    if (!entry.second) {
                        SHOW_ERROR("TM accepted \"" + entry.first + "\", but it should reject this input.");
                    }
                }  else if (interpreter.state() == Turing::Result::REJECT) {
                    if (entry.second) {
                        SHOW_ERROR("TM rejected \"" + entry.first + "\", but it should accept this input.");
                    }
                }
            }
        });
    }

    int expectResult(const Turing::Program& tm, const string& input, Turing::Result result) {
        vector<char32_t> inputVec;
        for (char ch: input) {
            inputVec.push_back(ch);
        }

        Turing::Interpreter interpreter(tm, inputVec);
        int steps = 0;
        for (; interpreter.state() == Turing::Result::RUNNING && steps < kMaxSteps; steps++) {
            interpreter.step();
        }

        /* We should be done by now. */
        if (interpreter.state() == Turing::Result::RUNNING) {
            SHOW_ERROR("TM still running after " + addCommasTo(kMaxSteps) + " steps on input \"" + input + "\". Possible infinite loop?");
        }
        /* Make sure we got the right result. */
        if (interpreter.state() == Turing::Result::ACCEPT && result == Turing::Result::REJECT) {
            SHOW_ERROR("TM should have rejected input \"" + input + "\", but accepted the input instead.");
        }
        if (interpreter.state() == Turing::Result::REJECT && result == Turing::Result::ACCEPT) {
            SHOW_ERROR("TM should have accepted input \"" + input + "\", but rejected the input instead.");
        }

        return steps;
    }
}

PROVIDED_TEST("PasswordChecker.tm") {
    ifstream input(kBaseDir + "PasswordChecker.tm", ios::binary);
    if (!input) SHOW_ERROR("Error opening file PasswordChecker.tm");

    Turing::Program tm(input);
    if (!tm.isValid()) SHOW_ERROR("TM contains errors and cannot be run; use the debugger to see what they are.");

    /* Make sure we accept 'quokka' and write down how long it took. */
    int acceptSteps = expectResult(tm, "quokka", Turing::Result::ACCEPT);

    /* Every other string must be rejected in exactly this many steps. */
    runPrivateTest("PasswordChecker.tm", [&](istream& in) {
        for (string line; getline(in, line); ) {
            int steps = expectResult(tm, line, Turing::Result::REJECT);
            if (steps != acceptSteps) {
                SHOW_ERROR("TM took " + pluralize(acceptSteps, "step") + " to accept \"quokka\", but rejected \"" + line + "\" in " + pluralize(steps, "step") + ".");
            }
        }
    });
}
